<?php
/**
 * BwTransifex Component
 *
 * BwTransifex language model class for the component frontend
 *
 * @version 1.0.1
 * @package BwTransifex
 * @subpackage BwTransifex Component Site
 * @author Romana Boldt
 * @copyright (C) 2025 Boldt Webservice <forum@boldt-webservice.de>
 * @support https://www.boldt-webservice.de/en/forum-en/forum/bwtransifex.html
 * @license GNU/GPL, see LICENSE.txt
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

namespace BoldtWebservice\Component\BwTransifex\Site\Model;

// No direct access
defined('_JEXEC') or die('Restricted access');

use BoldtWebservice\Component\BwTransifex\Administrator\Helper\BwTransifexHelperLanguage;
use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use stdClass;

/**
 * Define the BwTransifex project model class
 *
 * @package BwTransifex Site
 *
 * @since 1.0.0
 */
class LanguageModel extends BaseDatabaseModel
{
    /**
     * Constructor
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Populate State
     *
     * @return void
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    protected function populateState(): void
    {
        $app = Factory::getApplication();

        // Load state from the request.
        $pk = $app->input->getInt('zip');
        $this->setState('zip.id', $pk);
        $this->setState('project.id', $app->input->getInt('project'));
    }

    /**
     * Gets the languages from the db
     *
     * @return array
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function getLanguages(): array
    {
        $db = Factory::getContainer()->get('DatabaseDriver');
        $input = Factory::getApplication()->input;
        $id = $input->getInt('id', 0);
        $query = $db->getQuery(true);
        $params = ComponentHelper::getParams('com_bwtransifex');

        $query->select('*')->from('#__bwtransifex_zips')
            ->where($db->qn('project_id') . '=' . $db->q($id));

        // Include languages that are 0% complete?
        if (!$params->get('show_0', 1))
        {
            $query->where('completed > 0');
        }

        $db->setQuery($query);

        return $this->prepare($db->loadObjectList());
    }

    /**
     * Prepares the languages for output
     *
     * @param   array  $languages  - the array with the languages
     *
     * @return array
     *
     * @since 1.0.0
     */
    private function prepare(array $languages): array
    {
        if (count($languages))
        {
            foreach ($languages as $key => $language)
            {
                $iso = explode('-', $language->lang_name);

                if (is_array($iso) && count($iso))
                {
                    if (strlen($iso[0]) == 2)
                    {
                        $language->iso_lang_name = BwTransifexHelperLanguage::code2ToName($iso[0]);
                    }
                    elseif (strlen($iso[0]) == 3)
                    {
                        $language->iso_lang_name = BwTransifexHelperLanguage::code3ToName($iso[0]);
                    }

                    if ((isset($iso[1]) && strlen($iso[1]) == 2))
                    {
                        $language->iso_country_name = BwTransifexHelperLanguage::code2ToCountry($iso[1]);
                    }

                    // Prepare values for array_multi sort
                    $sort['iso_lang_name'][$key] = $language->iso_lang_name;
                    $sort['completed'][$key] = $language->completed;
                }
            }

            // Sort first on completed status and then on the real iso_lang_name
            array_multisort($sort['completed'], SORT_DESC, $sort['iso_lang_name'], SORT_ASC, $languages);
        }


        return $languages;
    }

    /**
     * Method to get an item.
     *
     * @param integer|null $pk The id of the item
     *
     * @return object
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function getItem(?int $pk = null): object
    {
        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);

        $query->select('*')->from('#__bwtransifex_zips')
            ->where('id=' . $db->q($this->getState('zip.id')));

        $db->setQuery($query);

        $item = $db->loadObject();

        if (!is_object($item)) {
            $item = new stdClass();
        }

        return $item;
    }

    /**
     * Get language resources for lang
     *
     * @return mixed
     *
     * @since 1.0.0
     */
    public function getResources()
    {
        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);

        $query->select('l.*, r.resource_name')->from('#__bwtransifex_languages AS l')
            ->leftJoin('#__bwtransifex_zips AS z ON z.lang_name = l.lang_name')
            ->leftJoin('#__bwtransifex_resources as r ON r.id = l.resource_id')
            ->where('z.id=' . $db->q($this->getState('zip.id')))
            ->where('l.project_id = ' . $db->q($this->getState('project.id')));

        $db->setQuery($query);

        return $db->loadObjectList();
    }
}
