<?php
/**
 * BwTransifex Component
 *
 * BwTransifex package model class for the component backend
 *
 * @version 1.0.1
 * @package BwTransifex
 * @subpackage BwTransifex Component Admin
 * @author Romana Boldt
 * @copyright (C) 2025 Boldt Webservice <forum@boldt-webservice.de>
 * @support https://www.boldt-webservice.de/en/forum-en/forum/bwtransifex.html
 * @license GNU/GPL, see LICENSE.txt
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

namespace BoldtWebservice\Component\BwTransifex\Administrator\Model;

use BoldtWebservice\Component\BwTransifex\Administrator\Helper\BwTransifexHelperPackage;
use Exception;
use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;

defined('_JEXEC') or die('Restricted access');

/**
 * Define the BwTransifex package model class
 *
 * @package BwTransifex Admin
 *
 * @since 1.0.0
 */
class PackageModel extends BaseDatabaseModel
{
    /**
     * @var int $projectId
     *
     * @since 1.0.0
     */
    private int $projectId = 0;

    /**
     * Constructor
     *
     * @param array $config - An array of configuration options (name, state, dbo, table_path, ignore_request).
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function __construct(array $config = array())
    {
        if (isset($config['project']))
        {
            $this->projectId = $config['project']->id;
        }

        parent::__construct($config);
    }

    /**
     * Add the lang pack to the table
     *
     * @param   $resources
     * @param string $language - the joomla lang
     *
     * @return void
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function add($resources, string $language): void
    {
        // Now add the current package
        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);
        $totalStrings = 0;
        $translatedStrings = 0;

        foreach ($resources as $resource)
        {
            $stats = json_decode($resource->raw_data);

            if ($stats)
            {
                $translatedStrings += (int) $stats->translated_strings;
                $totalStrings += (int) $stats->total_strings;
            }
        }

        if ($totalStrings > 0)
        {
            $langPackageVersion = BwTransifexHelperPackage::getNewLanguagePackageVersion($this->projectId, $language);

            $values = $db->q(0) .
                ',' . $db->q($this->projectId) .
                ',' . $db->q($language) .
                ',' . $db->q((int) (($translatedStrings / $totalStrings) * 100)) .
                ',' . $db->q(Factory::getDate()->toSql()) .
                ',' . $db->q($langPackageVersion);

            $query->insert('#__bwtransifex_zips');
            $query->columns(
                    array(
                        $db->qn('id'),
                        $db->qn('project_id'),
                        $db->qn('lang_name'),
                        $db->qn('completed'),
                        $db->qn('created_time'),
                        $db->qn('lang_version'),
                    )
                );
            $query->values($values);

            $db->setQuery($query);
            $db->execute();
        }
    }

    /**
     * Restore the language data to the table
     *
     * @param array $data
     *
     * @return void
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function restore(array $data): void
    {
        if ($data)
        {
            // Now add the resources
            $db = Factory::getContainer()->get('DatabaseDriver');
            $query = $db->getQuery(true);

            $values = $db->quote($data['project_id']) .
                ',' . $db->quote($data['lang_name']) .
                ',' . $db->quote($data['completed']) .
                ',' . $db->quote($data['created_time']) .
                ',' . $db->quote($data['lang_version']);

            $query->insert('#__bwtransifex_zips');
            $query->columns(
                    array(
                        $db->quoteName('project_id'),
                        $db->quoteName('lang_name'),
                        $db->quoteName('completed'),
                        $db->quoteName('created_time'),
                        $db->quoteName('lang_version'),
                    )
                );
            $query->values($values);

            $db->setQuery($query);
            $db->execute();
        }
    }

    /**
     * Count the resources of a project
     *
     * @return int
     *
     * @package BwTransifex
     *
     * @since 1.0.0
     */
    public function countResources(): int
    {
        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery('true');

        $query->select('COUNT(id) as count');
        $query->from('#__bwtransifex_resources');
        $query->where('project_id = ' . $db->q($this->projectId));

        $db->setQuery($query);

        return $db->loadObject()->count;
    }
}
