<?php
/**
 * BwTransifex Component
 *
 * BwTransifex language model class for the component backend
 *
 * @version 1.0.1
 * @package BwTransifex
 * @subpackage BwTransifex Component Admin
 * @author Romana Boldt
 * @copyright (C) 2025 Boldt Webservice <forum@boldt-webservice.de>
 * @support https://www.boldt-webservice.de/en/forum-en/forum/bwtransifex.html
 * @license GNU/GPL, see LICENSE.txt
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

namespace BoldtWebservice\Component\BwTransifex\Administrator\Model;

use BoldtWebservice\Component\BwTransifex\Administrator\Helper\BwTransifexHelperTransifex;
use Exception;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Log\Log;
use Joomla\CMS\MVC\Model\AdminModel;
use Joomla\CMS\Table\Table;
use Joomla\Database\DatabaseAwareInterface;
use Joomla\Database\DatabaseAwareTrait;
use stdClass;
use Throwable;

defined('_JEXEC') or die('Restricted access');

/**
 * Define the BwTransifex language model class
 *
 * @package BwTransifex Admin
 *
 * @since 1.0.0
 */
class LanguageModel extends AdminModel implements DatabaseAwareInterface
{
    use DatabaseAwareTrait;

    /**
     * @var int $projectId
     *
     * @since 1.0.0
     */
    private int $projectId = 0;

    /**
     * @var object|null $project
     *
     * @since 1.0.0
     */
    private ?object $project = null;

    /**
     * @var int $resourceId
     *
     * @since 1.0.0
     */
    private int $resourceId = 0;

    /**
     * Constructor
     *
     * @param array $config - array with config options
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function __construct(array $config = array())
    {
        if (isset($config['project']))
        {
            $this->projectId = $config['project']->id;
            $this->project   = $config['project'];
        }

        if (isset($config['resource']))
        {
            $this->resourceId = $this->getResource($config['resource']);
        }

        parent::__construct($config);
    }

    /**
     * Return a reference to a Table object, always creating it.
     *
     * @param string $name    The table type to instantiate
     * @param string $prefix  A prefix for the table class name. Optional.
     * @param array  $options Configuration array for model. Optional.
     *
     * @return        Table    A database object
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function getTable($name = 'Language', $prefix = 'Administrator', $options = array()): Table
    {
        return parent::getTable($name, $prefix, $options);
    }

    /**
     * Method for getting a form.
     *
     * @param array   $data     Data for the form.
     * @param boolean $loadData True if the form is to load its own data (default case), false if not.
     *
     * @return  Form|false    A JForm object on success, false on failure
     *
     * @throws Exception
     *
     * @since 1.0.0
     *
     */
    public function getForm($data = [], $loadData = true): bool|Form
    {
        // Get the form.
        $form = $this->loadForm('com_bwtransifex.project', 'Project',
            array('control' => 'jform', 'load_data' => $loadData));

        if (empty($form))
        {
            return false;
        }

        return $form;
    }

    /**
     * Add languages to table
     *
     * @param stdClass $languages - the languages info
     *
     * @return void
     *
     * @since 1.0.0
     */
    public function add(stdClass $languages): void
    {
        // Add the resources
        $config = parse_ini_string($this->project->transifex_config, true);

        foreach ($languages as $key => $language)
        {
            $langCode = BwTransifexHelperTransifex::getLangCode($key, $config);

            if ($langCode)
            {
                $data = array(
                    'project_id' => $this->projectId,
                    'resource_id' => $this->resourceId,
                    'lang_name' => $langCode,
                    'completed' => $language->completed,
                    'raw_data' => json_encode($language),
                );

                parent::save($data);
            }
        }
    }

    /**
     * Get resource information
     *
     * @param string $name - the resource name
     *
     * @return int
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    private function getResource(string $name): int
    {
        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);

        $query->select('id');
        $query->from('#__bwtransifex_resources');
        $query->where($db->qn('resource_name') . '=' . $db->q($name));
        $query->where($db->qn('project_id') . '=' . $db->q($this->projectId));
        $db->setQuery($query, 0, 1);

        return (int)$db->loadObject()->id;
    }

    /**
     * Get all resources for the language
     *
     * @param string $jlang - the joomla lang
     *
     * @return array
     *
     * @since 1.0.0
     */
    public function getResourcesForLang(string $jlang): array
    {
        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);

        $query->select(
            array(
                $db->qn('l.resource_id'),
                $db->qn('r.resource_name'),
                $db->qn('l.lang_name'),
                $db->qn('l.completed'),
                $db->qn('l.raw_data')
            )
        );
        $query->from('#__bwtransifex_languages AS l');
        $query->leftJoin('#__bwtransifex_resources AS r ON l.resource_id = r.id');
        $query->where($db->qn('l.lang_name') . '=' . $db->q($jlang));
        $query->where($db->qn('l.project_id') . '=' . $db->q($this->projectId));

        $db->setQuery($query);

        Log::addLogger(array('text_file' => 'com_bwtransifex.error.php'));
        Log::add('Query from getResourcesForLang: ' . $query);

        return (array)$db->loadObjectList();
    }

    /**
     * Get all language info for a resource the table languages knows
     *
     * @param string $jLang        - the joomla lang
     * @param string $resourceName - the resource name
     *
     * @return object
     *
     * @since 1.0.0
     */
    public function getLangInfo(string $jLang, string $resourceName): object
    {
        $response = array();
        $db       = Factory::getContainer()->get('DatabaseDriver');
        $query    = $db->getQuery(true);
        $langInfo = new stdClass();

        $query->select('*')->from('#__bwtransifex_languages as l');
        $query->leftJoin('#__bwtransifex_resources as r ON r.id = l.resource_id');
        $query->where($db->qn('r.resource_name') . '=' . $db->q($resourceName));
        $query->where($db->qn('l.project_id') . '=' . $db->q($this->projectId));
        $query->where($db->qn('l.lang_name') . '=' . $db->q($jLang));

        try
        {
            $db->setQuery($query, 0, 1);

            $langInfo = $db->loadObject();
        }
        catch (Throwable $e)
        {
            $response['message'] = $e->getMessage();
            $response['status'] = 'failure';
        }
        finally
        {
            if (!is_object($langInfo))
            {
                $langInfo = new stdClass();
            }
        }

        return $langInfo;
    }
}
