<?php
/**
 * BwTransifex Component
 *
 * BwTransifex transifex helper class for the component backend
 * *
 * BwTransifex is a largely reworked fork of Jonathan Daniel Dimitrov´s cTransifex package
 *
 * @version 1.0.1
 * @package BwTransifex
 * @subpackage BwTransifex Component Admin
 * @author Romana Boldt
 * @copyright (C) 2025 Boldt Webservice <forum@boldt-webservice.de>
 * @support https://www.boldt-webservice.de/en/forum-en/forum/bwtransifex.html
 * @license GNU/GPL, see LICENSE.txt
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

namespace BoldtWebservice\Component\BwTransifex\Administrator\Helper;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Log\Log;

defined('_JEXEC') or die('Restricted access');

/**
 * Define the BwTransifex transifex helper class
 *
 * @package BwTransifex Admin
 *
 * @since  1.0.0
 */
class BwTransifexHelperTransifex
{
    private static string $apiUrl = 'https://rest.api.transifex.com/';

    private static array $languages = array();

    /**
     * Gets data from transifex.com by curl
     *
     * The returned array provides
     * - data is the direct response of the request
     * - info contains information about the transfer (see return values at https://www.php.net/manual/de/function.curl-getinfo.php)
     *
     * @param string $txPath    endpoint and filter for request to transifex.com
     * @param string $curlData  needed for POST operations
     * @param string $curlMode  needed for other methods than GET or HEAD
     * @param bool   $rawUrl    when overriding normal behaviour is needed
     *
     * @return array
     *
     * @since 1.0.0
     */
    public static function getData(string $txPath, string $curlData = '', string $curlMode = '', bool $rawUrl = false): array
    {
        $token = ComponentHelper::getParams('com_bwtransifex')->get('tx_password');

        $header = array(
            'accept: application/vnd.api+json',
            'Authorization: Bearer ' . $token,
            'content-type: application/vnd.api+json',
        );

        $url = self::$apiUrl . $txPath;
        if ($rawUrl)
        {
            $url = $txPath;
        }

        $ch = curl_init();
        $timeout = 120;
        curl_setopt($ch, CURLOPT_HTTPHEADER, $header);
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, $timeout);
        curl_setopt($ch, CURLOPT_TIMEOUT, 400);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

        if ($curlData !== '')
        {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $curlData);
        }


        if ($curlMode !== '')
        {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $curlMode);
        }

        // Get the response data
        $data = curl_exec($ch);

        // Get info array about the request
        $info = curl_getinfo($ch);

        // Close the request
        curl_close($ch);

        return array('data' => $data, 'info' => $info);
    }

    /**
     * Get the langCode and transform it to the format we need
     * en-GB : en_GB or en_GB : en-GB
     *
     * @param string  $lang          - the lang
     * @param array   $projectConfig - the config
     * @param boolean $transifex     - true = joomla, false = transifex
     *
     * @return string
     *
     * @since 1.0.0
     */
    public static function getLangCode(string $lang, array $projectConfig, bool $transifex = false): string
    {
        $languages = self::getLangmap($projectConfig);

        if ($transifex)
        {
            $languages = array_flip($languages);
        }

        if (isset($languages[$lang]))
        {
            return $languages[$lang];
        }

        Log::addLogger(array('text_file' => 'com_bwtransifex.error.php'));
        Log::add('There is no lang map entry for ' . $lang, Log::ERROR);

        return '';
    }

    /**
     * Get the language map
     *
     * @param array $projectConfig - the project config
     *
     * @return array
     *
     * @since 1.0.0
     */
    private static function getLangmap(array $projectConfig): array
    {
        if (!count(self::$languages))
        {
            $languages       = array();
            $componentConfig = ComponentHelper::getParams('com_bwtransifex');
            $langMap         = explode(',', $componentConfig->get('tx_lang_map'));

            foreach ($langMap as $map)
            {
                $langCodes                      = explode(':', $map);
                $languages[trim($langCodes[0])] = trim($langCodes[1]);
            }

            if (isset($projectConfig['main']['lang_map']))
            {
                $langMap = explode(',', $projectConfig['main']['lang_map']);

                foreach ($langMap as $map)
                {
                    $langCodes                      = explode(':', $map);
                    $languages[trim($langCodes[0])] = trim($langCodes[1]);
                }
            }

            self::$languages = $languages;
        }

        return self::$languages;
    }
}
