<?php
/**
 * BwTransifex Component
 *
 * BwTransifex projects json controller class for the component backend
 *
 * BwTransifex is a largely reworked fork of Jonathan Daniel Dimitrov´s cTransifex package
 *
 * @version 1.0.1
 * @package BwTransifex
 * @subpackage BwTransifex Component Admin
 * @author Romana Boldt
 * @copyright (C) 2025 Boldt Webservice <forum@boldt-webservice.de>
 * @support https://www.boldt-webservice.de/en/forum-en/forum/bwtransifex.html
 * @license GNU/GPL, see LICENSE.txt
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

namespace BoldtWebservice\Component\BwTransifex\Administrator\Controller;

use BoldtWebservice\Component\BwTransifex\Administrator\Helper\BwTransifexHelperPackage;
use BoldtWebservice\Component\BwTransifex\Administrator\Helper\BwTransifexHelperTransifex;
use BoldtWebservice\Component\BwTransifex\Administrator\Libraries\BwWebApp;
use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\MVC\Controller\AdminController;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\CMS\Session\Session;
use RuntimeException;
use stdClass;
use Throwable;

defined('_JEXEC') or die('Restricted access');

/**
 * Define the BwTransifex project json controller class
 *
 * @package BwTransifex Admin
 *
 * @since   1.0.0
 */
class ProjectjsonController extends AdminController
{
    /**
     * @var object project_id
     *
     * @since 1.0.0
     */
    public object $project;

    /**
     * @var string $orgSlug
     *
     * @since 1.0.0
     */
    private string $orgSlug = '';

    /**
     * @var string $txProjectIdentifier
     *
     * @since 1.0.0
     */
    private string $txProjectIdentifier = '';

    /**
     * @var array $projectConfig
     *
     * @since 1.0.0
     */
    private array $projectConfig = [];

    /**
     * The constructor
     *
     * @param   array  $config  - the controller config
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function __construct(array $config = array())
    {
        parent::__construct($config);

        // We need some project data everywhere in the controller, so let's set it!
        $this->setProject();
        $this->setOrgSlug();
        $this->setProjectConfig();
        $this->setProjectIdentifier();
    }

    /**
     * Set the project
     *
     * @return void
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    private function setProject(): void
    {
        $projectModel  = $this->getModel('Project', 'BwTransifexModel');
        $projectId     = Factory::getApplication()->getInput()->getInt('project-id');
        $this->project = $projectModel->getItem($projectId);
    }

    /**
     * Set the organization slug
     *
     * @return void
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    private function setOrgSlug(): void
    {
        $this->orgSlug = ComponentHelper::getParams('com_bwtransifex')->get('tx_username', '');
    }

    /**
     * Set the organization slug
     *
     * @return void
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    private function setProjectConfig(): void
    {
        $this->projectConfig = parse_ini_string($this->project->transifex_config, true);
    }

    /**
     * Set the project identifier in the format
     * o:organization:p:project
     * where organization and project are the values needed to identify with transifex
     *
     * @return void
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    private function setProjectIdentifier(): void
    {
        $this->txProjectIdentifier = 'o:' . $this->orgSlug . ':p:' . $this->project->transifex_slug;
    }

    /**
     * Proxy for getModel.
     *
     * @param string $name   The name of the model.
     * @param string $prefix The prefix for the PHP class name.
     * @param array  $config An optional associative array of configuration settings.
     *
     * @return BaseDatabaseModel
     *
     * @since    1.0.0
     */
    public function getModel($name = 'Project', $prefix = 'Administrator', $config = array('ignore_request' => true))
    {
        return parent::getModel($name, $prefix, $config);
    }

    /**
     * Get the project resources of a project known by transifex.com
     *
     * This method is called by ajax and does a curl to transifex.com
     *
     * Because this is a WebApp the method has no return value, nevertheless it returns the response of the curl to the
     * calling ajax as json encoded array
     * - message
     * - status
     *
     * On success the message contains a list (of objects?) of the known resources
     *
     * @return void
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function resources(): void
    {
        $this->checkSession();

        $appWeb = new BwWebApp();
        $appWeb->setHeader('Content-Type', 'application/json', true);

        $response = array();
        $endpoint = 'resources';

        $projectFilter = '?filter[project]=' . $this->txProjectIdentifier;
        $txFilter      = $endpoint . $projectFilter;

        try
        {
            $resources = BwTransifexHelperTransifex::getData($txFilter);

            if (isset($resources['info']) && $resources['info']['http_code'] != 200)
            {
                $error = json_decode($resources['data'], true)['errors'][0];
                $message = json_encode($error);
                $response['message'] = $message;
                $response['status'] = 'failure';

            }
            else
            {
                $resources = json_decode($resources['data']);

                foreach ($resources->data as $resource)
                {
                    $response['message'][] = $resource->attributes->slug;
                }

                // If we have resources add them to the db, not before cleaning tables
                if (is_array($response['message']))
                {
                    $resourceModel = $this->getModel('Resource', 'BwTransifexModel', array('project_id' => $this->project->id));

                    $resourceModel->add($response['message']);
                }

                $response['status'] = 'success';
            }
        }
        catch (Throwable $e)
        {
            $response['message'] = $e->getMessage();
            $response['status'] = 'failure';
        }
        finally
        {
            echo json_encode($response);
            $appWeb->close();
        }
    }

    /**
     * Get language stats per resource of a project, called by ajax
     *
     Get the project resources of a project known by transifex.com
     *
     * This method is called by ajax and does a curl to transifex.com
     *
     * Because this is a WebApp the method has no return value, nevertheless it returns the response of the curl to the
     * calling ajax as json encoded array
     * - message
     * - status
     *
     * @return void
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function languageStats(): void
    {
        $this->checkSession();

        $response = array();
        $endpoint = 'resource_language_stats';

        try
        {
            $resource       = Factory::getApplication()->input->getString('resource');
            $projectFilter  = '?filter[project]=' . $this->txProjectIdentifier;
            $resourceFilter = '';

            $txFilter = $endpoint . $projectFilter . $resourceFilter;

            $txData = BwTransifexHelperTransifex::getData($txFilter);

            $responseData = array();

            if (isset($txData['info']) && $txData['info']['http_code'] == 200)
            {
                if (!$txData['data'])
                {
                    throw new Exception(Text::_('COM_BWTRANSIFEX_PROJECT_ERROR_CURL_FALSE'));
                }

                $stats = get_object_vars(json_decode($txData['data']));

                foreach ($stats['data'] as $stat)
                {
                    $currentResource = explode(':l:', explode(':r:', $stat->id)[1])[0];
                    $currentLang     = explode(':l:', $stat->id)[1];
                    $sourceLang      = $this->getSourceLang($resource);

                    if ($currentLang !== $sourceLang)
                    {
                        $dataToStore  = new stdClass();
                        $currentStats = $stat->attributes;
                        $completed    = ((float)$currentStats->translated_strings / (float)$currentStats->total_strings) * 100;
                        $currentStats->completed = $completed;

                        $dataToStore->$currentLang = $currentStats;
                        $responseData[] = $currentLang;

                        $languageModel = $this->getModel('Language', 'BwTransifexModel', array('project' => $this->project, 'resource' => $currentResource));
                        $languageModel->add($dataToStore);
                    }
                }

                $response['status'] = 'success';
                $response['message'] = $responseData;
            }
            else
            {
                $response['status']  = 'failure';
                $response['message'] = $txData['data'];
            }
        }
        catch (RuntimeException | Exception $e)
        {
            $response['status']  = 'failure';
            $response['message'] = $e->getMessage();
        }
        finally
        {
            echo json_encode($response);
            jexit();
        }
    }

    /**
     * Download translations for all resources for a language and zip them, called by ajax
     *
     * @return void
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function langpack(): void
    {
        $session = Factory::getApplication()->getSession();
        $this->checkSession();
        Log::addLogger(array('text_file' => 'com_bwtransifex.error.php'));

        // Set some defaults
        $minPercent = 0;
        $response   = array();

        // Get the minPercent variable from main project config
        if (isset($this->projectConfig['main']['minimum_perc']))
        {
            $minPercent = $this->projectConfig['main']['minimum_perc'];
        }

        // Get the source language
        $srcLang = $this->getSourceLang('');

        try
        {
            // Get the desired language from input
            $lang  = Factory::getApplication()->input->getString('language');
            $jLang = BwTransifexHelperTransifex::getLangCode($lang, $this->projectConfig);

            // Check if we have to update the language packages
            $updateNeeded = $this->checkForNeededUpdate($jLang);
            $session->set('updateNeeded', $updateNeeded, 'bwtransifex');


            Log::add('SrcLang: ' . $srcLang . ' transmitted lang: ' . $lang . ' JLang: ' . $jLang);

            // Getting the source language is not needed and causes problems in the further cause
            if (($jLang != '') && ($jLang !== $srcLang))
            {
                // Get all resources for this project language
                $languageModel = $this->getModel('Language', 'BwTransifexModel', array('project' => $this->project));
                $resources     = $languageModel->getResourcesForLang($jLang);

                // Iterate over these resources to get the files
                foreach ($resources as $resource)
                {
                    $langInfo = $languageModel->getLangInfo($jLang, $resource->resource_name);

                    // Check if we have a minPercent for this resource (if not use the main Perc)
                    if (isset($this->projectConfig[$this->txProjectIdentifier . ':r:' . $resource->resource_name]['minimum_perc']))
                    {
                        $minPercent = $this->projectConfig[$this->txProjectIdentifier . ':r:' . $resource->resource_name]['minimum_perc'];
                    }

                    // Download the file only if min percent don't block it
                    if ($minPercent == 0 || (property_exists($langInfo, 'completed') && $minPercent < $langInfo->completed))
                    {
                        // Get the language file from transifex-com for this resource
                        if (!$this->langFile($resource->resource_name, $lang))
                        {
                            $errorMsg = Text::sprintf('COM_BWTRANSIFEX_PROJECT_ERROR_LANGPACK_DOWNLOAD', $jLang, $resource->resource_name);

                            Log::add($errorMsg, Log::ERROR);

                            throw new Exception($errorMsg);
                        }
                    }
                }

                $response = BwTransifexHelperPackage::package($jLang, $this->project);

                if ($response['status'] === 'success')
                {
                    $packageModel = $this->getModel('Package', 'BwTransifexModel', array('project' => $this->project));
                    $langPackVersions = $session->get('prevLangPackVersions', null, 'bwtransifex');

                    if ($response['action'] === 'changed' || is_null($langPackVersions[$jLang]))
                    {
                        $packageModel->add($resources, $jLang);
                    }
                    else
                    {
                        $packageModel->restore($langPackVersions[$jLang]);
                    }
                }
                else
                {
                    Log::add('We could not package ' . $jLang, Log::ERROR);

                    $response['status']  = 'failure';

                    if (key_exists('data', $response))
                    {
                        $response['message'] = $response['data'];
                    }
                }
            }
            else
            {
                if ($jLang === $srcLang)
                {
                $response['status']  = 'warning';
                    $response['message'] = Text::sprintf('COM_BWTRANSIFEX_PROJECT_WARNING_SOURCE_LANG_SKIPPED', $jLang);
                }
                else
                {
                    $response['status']  = 'failure';
                    $response['message'] = Text::sprintf('COM_BWTRANSIFEX_PROJECT_ERROR_LANGPACK_NOT_SPECIFIED', $lang);
                }
            }
        }
        catch (RuntimeException | Exception $e)
        {
            $response['status'] = 'failure';
            $response['message'] = $e->getMessage();
        }
        finally
        {
            echo json_encode($response);
            jexit();
        }
    }

    /**
     * Get the language files from transifex.com
     *
     * @param string $resource - the resource name
     * @param string $lang     - the lang name
     *
     * @return bool
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function langFile(string $resource, string $lang): bool
    {
        $this->checkSession();

        // See https://developers.transifex.com/reference/post_resource-translations-async-downloads
        $endpoint           = 'resource_translations_async_downloads';
        $resourceIdentifier = $this->txProjectIdentifier . ':r:' . $resource;

        // Construct the download resourceIdentifier
        $attributes = array(
            'callback_url' => null,
            'content_encoding' => 'text',
            'file_type' => 'default',
            'mode' => 'default',
        );

        $relationData = array(
            'type' => 'resources',
            'id'   => $resourceIdentifier,
        );

        $languageData = array(
            'type' => 'languages',
            'id' => 'l:' . $lang,
        );

        $curlData = array(
            'data' => array(
                'attributes' => $attributes,
                'relationships' => array(
                    'resource' => array(
                        'data' => $relationData,
                    ),
                    'language' => array(
                        'data' => $languageData,
                    ),
                ),
                'type' => $endpoint,
            )
        );

        // Start building download file at transifex.com
        $txPath      = $endpoint;
        $fileRequest = BwTransifexHelperTransifex::getData($txPath, json_encode($curlData));

        if (isset(json_decode($fileRequest['data'])->errors))
        {
            return json_decode($fileRequest['data'])->errors;
        }

        $downloadStatus = json_decode($fileRequest['data'])->data->attributes->status;

        if (!$downloadStatus)
        {
            $downloadStatus = '';
        }

        // Get file id to download
        $txPath .= '/' . json_decode($fileRequest['data'])->data->id;

        $loopCounter  = 0;
        $sleepingTime = 1;

        // Loop to check if building download file has finished
        while (strtolower($downloadStatus) !== 'success')
        {
            // Check the response of the with the provided ID enhanced request
            $fileDownload = BwTransifexHelperTransifex::getData($txPath, '', "GET");

            // If returned status is 303, the building of the file to download has completed and the file can be
            // downloaded at the returned redirect_url
            if ($fileDownload['info']['http_code'] === 303)
            {
                $txPath = $fileDownload['info']['redirect_url'];
                $file   = BwTransifexHelperTransifex::getData($txPath, '', "GET", true);

                if ($file['info']['http_code'] === 200)
                {
                    $downloadStatus = 'success';
                    $sleepingTime   = 0;
                }
            }

            if ($loopCounter++ > 5)
            {
                break;
            }

            sleep($sleepingTime);
        }

        // Building download file has finished (or loop above exited without success)
        if (isset($file['info']) && $file['info']['http_code'] == 200)
        {
            if (isset($this->projectConfig[$resourceIdentifier]))
            {
                $jlang = BwTransifexHelperTransifex::getLangCode($lang, $this->projectConfig);

                // Save the downloaded file locally
                return BwTransifexHelperPackage::saveLangFile($file, $jlang, $this->project, $resource, $this->projectConfig);
            }
        }

        return false;
    }

    /**
     * Checks the user session
     *
     * @return void
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    private function checkSession(): void
    {
        if (!Session::checkToken())
        {
            $response['status'] = 'failure';
            $response['message'] = Text::_('JINVALID_TOKEN');

            echo json_encode($response);
            jexit();
        }
    }

    /**
     * Get the source language for a resource in transifex format (with underscore)
     *
     * First get the value from component
     * Second get the value from the project, if provided
     * Third get the value from the resource, if provided
     *
     * @param string $resource
     *
     * @return string
     *
     * @since 1.0.0
     */
    private function getSourceLang(string $resource): string
    {
        // Get source language from component settings
        $srcLang = ComponentHelper::getParams('com_bwtransifex')->get('tx_source_lang', 'en-GB');

        // Get source language from project settings if provided
        if (isset($this->projectConfig['main']['source_lang']))
        {
            $srcLang = $this->projectConfig['main']['source_lang'];
        }

        // Get source language from resource if provided
        if ($resource !== '')
        {
            $resourceSlug = $this->txProjectIdentifier . ':r:' . $resource;

            if (isset($this->projectConfig[$resourceSlug]['source_lang']))
            {
                $srcLang = $this->projectConfig[$resourceSlug]['source_lang'];
            }
        }

        return $srcLang;
    }

    /**
     * Check if we have to create a new language package.
     *
     * This is always the case except
     * - there is no request from the user by parameter 'update_always',
     * - and there is no new project version
     * - and there are no changed language strings
     *
     * @param   string  $jLang
     *
     * @return bool
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    private function checkForNeededUpdate(string $jLang):bool
    {
        // If setting 'update_always' is set, we have to create new language packages, even if there are no new or
        // changed language stings or there is no new project version
        if ($this->project->params['update_always'])
        {
            return true;
        }

        $session = Factory::getApplication()->getSession();

        $langPackVersions = $session->get('prevLangPackVersions', null, 'bwtransifex');

        // If there is no old version for this language, create new language package
        if (!key_exists($jLang, $langPackVersions))
        {
            return true;
        }

        if (!is_array($langPackVersions[$jLang]) || !key_exists('lang_version', $langPackVersions[$jLang]))
        {
            return true;
        }

        // If there is a new project version, we have to create new language packages, although there are no new or
        // changed language strings
        $previousVersion  = $langPackVersions[$jLang]['lang_version'];

        if (version_compare($this->project->params['project_version'], $previousVersion, 'gt'))
        {
            return true;
        }

        // If there are new or changed language strings, we have to create a new language package for this language
        // Get previous raw_data (translation status) for this language
        $previousTxUpdate = $session->get('previousTxUpdate', null, 'bwtransifex');

        // Get current raw_data (translation status) for this language
        $currentTxUpdate = BwTransifexHelperPackage::getTxStatusFromDb($this->project->id, array($jLang));

        // Calculate changed strings since last update
        if ((!is_null($previousTxUpdate))
            && array_key_exists($jLang, $previousTxUpdate)
            && (!empty($currentTxUpdate)))
        {
            return BwTransifexHelperPackage::hasChangedStrings($previousTxUpdate[$jLang], $currentTxUpdate[$jLang]);
        }

        return true;
    }
}
