<?php
/**
 * BwTransifex Component
 *
 * BwTransifex list model class for the component frontend
 *
 * @version 1.0.1
 * @package BwTransifex
 * @subpackage BwTransifex Component Site
 * @author Romana Boldt
 * @copyright (C) 2025 Boldt Webservice <forum@boldt-webservice.de>
 * @support https://www.boldt-webservice.de/en/forum-en/forum/bwtransifex.html
 * @license GNU/GPL, see LICENSE.txt
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

namespace BoldtWebservice\Component\BwTransifex\Site\Model;

// No direct access
defined('_JEXEC') or die('Restricted access');

use Exception;
use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\CMS\Table\Table;

/**
 * Define the BwTransifex projects model class
 *
 * @package BwTransifex Site
 *
 * @since 1.0.0
 */
class ProjectsModel extends ListModel
{
    /**
     * The database object
     *
     * @var	?object
     *
     * @since       1.0.0
     */
    protected ?object $db = null;

    /**
     * The query object
     *
     * @var	object
     *
     * @since       1.0.0
     */
    protected $query;

    /**
     * Constructor
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Returns a Table object, always creating it.
     *
     * @param	string $name    The table type to instantiate
     * @param	string $prefix  A prefix for the table class name. Optional.
     * @param	array  $options Configuration array for model. Optional.
     *
     * @return	Table	A database object
     *
     * @throws Exception
     *
     * @since  1.0.0
     */
    public function getTable($name = 'Project', $prefix = 'Administrator', $options = array()): Table
    {
        return parent::getTable($name, $prefix, $options);
    }

    /**
     * Build an SQL query to load the list data.
     *
     * @throws Exception
     * @since    1.0.0
     */
    protected function getListQuery()
    {
        // Create a new query object.
        $this->db    = Factory::getContainer()->get('DatabaseDriver');
        $this->query = $this->db->getQuery(true);

        // Select the required fields from the table.
        $this->query->select(
            'p.id, p.title, p.description, p.params, p.alias,' .
            'p.published, p.access, p.created_time, p.created_by, p.ordering'
        );
        $this->query->from('#__bwtransifex_projects AS p');

        // Join over the asset groups.
        $this->query->select('ag.title AS access_level');
        $this->query->join('LEFT', '#__viewlevels AS ag ON ag.id = p.access');

        $this->getQueryWhere();
        $this->getQueryOrder();

        return $this->query;
    }

    /**
     * Method to build the MySQL query 'where' part
     *
     * @return 	void
     *
     * @throws Exception
     *
     * @since   1.0.0
     */
    private function getQueryWhere(): void
    {
        $this->getFilterByAccessLevel();
        $this->getFilterByPublished();
    }

    /**
     * Method to build the MySQL query 'order' part
     *
     * @return 	void
     *
     * @since   1.0.0
     */
    private function getQueryOrder(): void
    {
        $db        = $this->getDatabase();
        $orderCol  = $this->state->get('list.ordering', 'p.title');
        $orderDirn = $this->state->get('list.direction', 'asc');

        //sqlsrv change
        if ($orderCol == 'modified_time')
        {
            $orderCol = 'p.modified_time';
        }

        if ($orderCol == 'p.title')
        {
            $orderCol = 'p.title';
        }

        $this->query->order($db->quoteName($db->escape($orderCol)) . ' ' . $db->escape($orderDirn));
    }

    /**
     * Method to get the where part for access level
     *
     * @return 	void
     *
     * @throws Exception
     *
     * @since   1.0.0
     */
    private function getFilterByAccessLevel(): void
    {
        $groups = Factory::getApplication()->getIdentity()->getAuthorisedViewLevels();

        $this->query->whereIn($this->db->quoteName('p.access'), $groups);
    }

    /**
     * Method to get the where part for published
     *
     *
     * @return    void
     *
     * @throws Exception
     *
     * @since   1.0.0
     */
    private function getFilterByPublished(): void
    {
        $isRoot = Factory::getApplication()->getIdentity()->authorise('core.admin');

        if (!$isRoot)
        {
            $this->query->where($this->db->quoteName('p.published') . ' = 1');
        }
    }

    /**
     * Method to get a list of projects.
     *
     * Overridden to inject convert the attributes field into a Parameter object.
     *
     * @return	array|null	An array of objects on success, false on failure.
     *
     * @throws Exception
     *
     * @since	1.0.0
     */
    public function getItems(): ?array
    {
        $items	= parent::getItems();

        if (!is_countable($items)) {
            $items = array();
        }

        return $items;
    }
}
