/**
 * BwTransifex Component
 *
 * BwTransifex JavaScript for updating language files at transifex
 *
 * @version 1.0.1
 * @package BwTransifex
 * @subpackage BwTransifex Component Admin
 * @author Romana Boldt
 * @copyright (C) 2025 Boldt Webservice <forum@boldt-webservice.de>
 * @support https://www.boldt-webservice.de/en/forum-en/forum/bwtransifex.html
 * @license GNU/GPL, see LICENSE.txt
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */


const options = Joomla.getOptions('bwtx_project_refresh');
// let startTime = performance.now();
let data = {};

if (document.readyState === "loading") {
    // Loading hasn't finished yet
    document.addEventListener("DOMContentLoaded", prepareModal);
} else {
    // `DOMContentLoaded` has already fired
    prepareModal();
}

/**
 * Do preparations for the modal window and add event listener to the action button
 *
 * @since 1.0.0
 */
function prepareModal() {
    const links = document.getElementsByClassName('select-link');

    for(let i = 0; i < links.length; i += 1) {
        links[i].addEventListener('click', event => {
            options.projectId = event.currentTarget.getAttribute('data-id');
        })
    }

    const myModal = document.getElementById('bwtransifex-project-data');

    // Set some needed values
    sessionStorage.setItem('project-id', options.projectId);
    data = {
        'project-id': options.projectId,
        'token'     : options.token,
        'error'     : false
    }

    // Do the preparations
    checkForSessionStorage();
    prepareMessageContainer();
    refreshActionEvent();

    // Add event listener modal opened
    myModal.addEventListener('shown.bs.modal', () => {
        // Set some needed values
        options.baseUrl = document.getElementById('baseUrl').value;
        options.token   = document.getElementById('token').value;
        data = {
            'project-id': options.projectId,
            'token'     : options.token,
            'error'     : false
        }

        // Do the preparations
        checkForSessionStorage();
        prepareMessageContainer();
        refreshActionEvent();
    });

    // Add event listener modal closed
    myModal.addEventListener('hidden.bs.modal', () => {
        prepareMessageContainer();
        doTasksOnCloseModal();
    });
}

/**
 * Check if sessionStorage is available and do alert if not
 *
 * @since 1.0.0
 */
function checkForSessionStorage() {
    if (!window.sessionStorage) {
        alert(Joomla.Text._('COM_BWTRANSIFEX_ERROR_NO_SESSION_STORAGE'));
    }
}

/**
 * Empty the message container, if it exists. Otherwise, remove the second nested div from form.
 *
 * @since 1.0.0
 */
function prepareMessageContainer() {
    let messageContainer = document.getElementById('bwtx_message_container');

    if (messageContainer !== null) {
        messageContainer.innerHTML = "";
    }
    else {
        let formDivElements = document.evaluate("//*/form[@class='bwtransifex-project-data-form']/div/div", document, null, XPathResult.ORDERED_NODE_ITERATOR_TYPE, null);

        let thisDiv = formDivElements.iterateNext();
        let alertText = "Form div div in this document are:\n";
        while (thisDiv) {
            alertText += `${thisDiv.textContent}\n`;
            thisDiv.parentNode.removeChild(thisDiv);
            thisDiv = formDivElements.iterateNext();
        }

        createMessageContainer();
    }
}

/**
 * Create the message container.
 *
 * @since 1.0.0
 */
function createMessageContainer() {
    const outerMessageContainer = document.getElementById('bwtx_message_container-outer');
    let messageContainer = document.getElementById('bwtx_message_container');

    if (messageContainer === null) {
        messageContainer = document.createElement('div');
        messageContainer.id = 'bwtx_message_container';
        messageContainer.className = 'modal-message-container';
        messageContainer.setAttribute('data-test', 'project-modal-body-messages');
        outerMessageContainer.append(messageContainer);
    }
}

/**
 * Create a message for the message container.
 *
 * @since 1.0.0
 */
function createMessage (message, type, additional) {
    let messageContainer = document.getElementById('bwtx_message_container');


    let htmlElement = document.createElement('div');
    htmlElement.className = "bwtx_modal_message bwtx_modal_message_" + type;
    htmlElement.innerHTML = message;
    htmlElement.dataset.test = 'project-modal-body-' + additional;
    htmlElement.setAttribute('data-test', 'project-modal-body-message-single-' + additional);
    messageContainer.append(htmlElement);
}

/**
 * Do cleanup on modal close
 *
 * @since 1.0.0
 */
function doTasksOnCloseModal() {
    // Empty request queue
    Request.requestQueue = [];

    sessionStorage.removeItem('project-id');
    sessionStorage.removeItem('token');
}

/**
 * Remove current action event and add it anew to ensure, action is processed only once
 *
 * @since 1.0.0
 */
function refreshActionEvent() {
    // Remove close events
    const actionButton = document.getElementById('bw_action');
    // const resourceRequest = doResourceRequest();

    actionButton.removeEventListener('click', doResourceRequest, false);

    // Do ajax request for resources
    actionButton.addEventListener('click', doResourceRequest, false);
}

/**
 * Remove current action event and add it anew to ensure, action is processed only once
 *
 * @since 1.0.0
 */
function switchActionButtons(displayType) {
    const actionButtons = document.getElementsByClassName('bw_action_buttons');

    if (actionButtons.length > 0) {
        for (let i = 0; i < actionButtons.length; i++){
            actionButtons[i].style.visibility = displayType;
        }
    }
}

/**
 * Method to output the error message
 *
 * @since 1.0.0
 */
function outputErrorMessage(error) {
    let message = Joomla.Text._("COM_BWTRANSIFEX_MODAL_ERROR_HEADING");
    message = message + Joomla.Text._("COM_BWTRANSIFEX_MODAL_ERROR_STATUS") + error.status + '<br />';
    message = message + Joomla.Text._("COM_BWTRANSIFEX_MODAL_ERROR_CODE") + error.code + '<br />';
    message = message + Joomla.Text._("COM_BWTRANSIFEX_MODAL_ERROR_TITLE") + error.title + '<br />';
    message = message + Joomla.Text._("COM_BWTRANSIFEX_MODAL_ERROR_DETAIL") + error.detail + '<br />';

    // Output message
    createMessage(message, 'error', 'error');
}

    /**
 * This is the first task/request to trigger the complete chain:
 * Get the resources for the project from transifex
 *
 * This is a single request
 *
 * A named function is needed, because the event listener has first to be removed to ensure that the action is processed
 * only once.
 *
 * See (see <joomla_root>/administrator/components/com_bwtransifex/src/Controller/ProjectjsonController.php public function resources())
 *
 * @since 1.0.0
 */
function doResourceRequest() {
    // startTime = performance.now();
    prepareMessageContainer();

    // Remove close buttons while running to prevent processing interruption
    switchActionButtons('hidden');

    // Build and output start message
    let message = Joomla.Text._('COM_BWTRANSIFEX_RESOURCES_FETCH');
    createMessage(message, 'heading', 'resource-start');

    // The action should be projectjson.resources…
    // (see <joomla_root>/administrator/components/com_bwtransifex/src/Controller/ProjectjsonController.php Line 191)
    let action = document.getElementById('bw_form_action').value + options.projectId;
    action = action + '&' + options.token + '=1';

    // Make the initial request to get the resources of the project
    // See <joomla_root>/media/system/js/core.es6.js
    Joomla.request({
        url      : action,
        method   : 'POST',
        data     : null,
        perform  : true,
        headers  : {
            'Content-Type': 'application/x-www-form-urlencoded'
        },
        onComplete: function onComplete(xhr) {
            try {
                const response = JSON.parse(xhr.response);
                const resources = response.message;

                // On success store the result (resources) to a session variable and go on
                if (response.status === 'success') {
                    sessionStorage.setItem('resources', JSON.stringify(resources));

                    let message = Joomla.Text._('COM_BWTRANSIFEX_RESOURCES_FOUND');
                    createMessage(message, 'heading', 'resource-found');

                    for (let i = 0; i < resources.length; i++) {
                        message = resources[i];
                        createMessage(message, 'notify', message);
                    }

                    languageStats();
                }
                // On error show message and exit
                else {
                    let eMessage = {};
                    const error = JSON.parse(response.message)
                    eMessage.status = error.status;
                    eMessage.code   = error.code;
                    eMessage.title  = error.title;
                    eMessage.detail = error.detail;
                    outputErrorMessage(eMessage, 'error', 'error')
                    doFinalization(true)
                }
            } catch (e) {
                outputErrorMessage(e.message);
                doFinalization(true);
            }
        },
        onError  : function onError(xhr) {
            let message = JSON.parse(xhr.responseText).message;
            alert('Error: \n' + message);
        }
    });
}

/**
 * Get the language stats, which exists at transifex, per resource
 *
 * This is a single request
 *
 * See (see <joomla_root>/administrator/components/com_bwtransifex/src/Controller/ProjectjsonController.php public function languageStats())
 *
 * @since 1.0.0
 * */
function languageStats() {
    // Build and output start message
    let message = Joomla.Text._("COM_BWTRANSIFEX_LANGSTATS_FETCH");
    createMessage(message, 'heading', 'language-stats-start');

    const resources = JSON.parse(sessionStorage.getItem('resources'));
    const projectId = options.projectId;
    let url = 'index.php?option=com_bwtransifex&task=projectjson.languageStats&format=json&project-id=' + options.projectId;
    url = url + '&' + options.token + '=1&resource=';

    // Loop over the resources
    if (resources.length) {
        let resource = resources[0];

        let data = {
            'token'     : options.token,
            'project-id': projectId,
            'resource'  : resource
        }

        Joomla.request({
            url    : url + resource,
            method : 'POST',
            data   : data,
            promise: true,
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded'
            },
        }).then((xhr) => {
            try {
                const response     = JSON.parse(xhr.response);
                const responseData = response.message;

                if (response.status === 'success') {
                    let uniqueData = [...new Set(responseData)];
                    sessionStorage.setItem('languages', JSON.stringify(uniqueData));

                    // let message = "We have found the following languages for the resource " + resource + ':';
                    let message = Joomla.Text._("COM_BWTRANSIFEX_LANGSTATS_FOUND");
                    createMessage(message, 'heading', 'language-stats-found-for-' + resource);

                    for (let i = 0; i < uniqueData.length; i++) {
                        message = uniqueData[i];
                        createMessage(message, 'notify', 'language-stats-' + message + '-found-for-' + resource);
                    }
                    getLanguageFiles();
                } else {
                    // Output error message
                    createMessage(responseData, 'error');
                    throw new Error(responseData);
                }
            }
            catch (e) {
                let error = true;
                if (e.hasOwnProperty('status')) {
                    outputErrorMessage(e.message);
                    error = false;
                } else {
                    let eMessage = {};
                    eMessage.status = e.fileName;
                    eMessage.code   = e.lineNumber;
                    eMessage.title  = e.message;
                    eMessage.detail = xhr.responseText;
                    outputErrorMessage(eMessage);
                }
                doFinalization(error);
            }
        })
    }
}

/**
 * Get the translated language files per language
 *
 * These are multiple requests in parallel, one for each language
 *
 * See (see <joomla_root>/administrator/components/com_bwtransifex/src/Controller/ProjectjsonController.php public function langpack())
 *
 * @since 1.0.0
 */
function getLanguageFiles() {
    const langs = JSON.parse(sessionStorage.getItem('languages'));
    // Build and output start message
    let message = Joomla.Text._("COM_BWTRANSIFEX_LANGPACK_FETCH");
    createMessage(message, 'heading', 'language-file-start');

    // const endTime1 = performance.now();
    // const time = (endTime1 - startTime) / 1000;

    // createMessage('Until now it took ' + time + 'seconds', 'notify', '');

    const projectId = sessionStorage.getItem('project-id');
    let url = 'index.php?option=com_bwtransifex&task=projectjson.langpack&format=json&project-id=' + options.projectId;
    url = url + '&' + options.token + '=1&language=';

    let itemsProcessed = 0;

    langs.sort().forEach(language => {
        let data = {
            'token'     : options.token,
            'project-id': projectId,
            'language'  : language
        }

        // Build and output start message
        let message = Joomla.Text._("COM_BWTRANSIFEX_LANGPACK_LANG") + language;
        createMessage(message, 'notify', 'language-files-start-for-' + language);

        Joomla.request({
            url   : url + language,
            method: 'POST',
            data  : data,
            // promise: true,
            perform  : true,
            headers  : {
                'Content-Type': 'application/x-www-form-urlencoded'
            },
            onSuccess: (xhr) => {
                try {
                    const response = JSON.parse(xhr);

                    if (response.status === 'success') {
                        // sessionStorage.setItem(resource, data);

                        let message = Joomla.Text._("COM_BWTRANSIFEX_LANGPACK_CREATE") + language.replaceAll('_', '-') + ':';
                        createMessage(message, 'heading', 'language-generate-zip-for-' + language);
                        createMessage(response.message, 'notify', 'language-zip-generated-for-' + language);
                    } else {
                        if (response.status === 'failure') {
                            // Build error message for php errors
                            message = response.message;

                            // Output message
                            createMessage(message, 'error', 'error');
                        } else if (response.status === 'warning') {
                            // Build error message for php errors
                            message = response.message;

                            // Output message
                            createMessage(message, 'warning', 'warning');
                        } else {
                            // Build message for ajax errors
                            outputErrorMessage(JSON.parse(xhr.response).message)
                        }
                    }
                }
                catch (e) {
                    const rawMessage = xhr.split('{')
                    const problem = rawMessage[0]

                    if (problem !== '') {
                        createMessage(problem, 'error');
                    } else {
                        createMessage(xhr, 'error', 'error');
                    }
                }

                itemsProcessed++;
                if (itemsProcessed === langs.length) {
                    doFinalization(false);
                }
            },
            onError  : (xhr) => {
                let message = xhr.responseText;
                createMessage(message, 'error', 'error');
            }
        });
    })
}

/*
 * Do finalization:
 * - Show action buttons
 * - cleanup storage
 * - output finished message
 *
 * @since 1.0.0
 */
function doFinalization(error) {
    // Show action button when processing has finished
    switchActionButtons('visible');

    // Cleanup session storage
    sessionStorage.clear();

    // Build message

    // Output message
    if (error) {
        let message = Joomla.Text._("COM_BWTRANSIFEX_LANGPACK_FINALIZATION_ERROR");
        createMessage(message, 'heading', 'zip-creation-finished');
    }
    else {
        let message = Joomla.Text._("COM_BWTRANSIFEX_LANGPACK_FINALIZATION");
        createMessage(message, 'heading', 'zip-creation-finished');
    }
    // const endTime = performance.now();
    // const time = (endTime - startTime) / 1000;

    // createMessage('Updating language files took ' + time + 'seconds', '', '');
}
