<?php
/**
 * BwTransifex Component
 *
 * BwTransifex table class for the component backend
 *
 * @version 1.0.1
 * @package BwTransifex
 * @subpackage BwTransifex Component Admin
 * @author Romana Boldt
 * @copyright (C) 2025 Boldt Webservice <forum@boldt-webservice.de>
 * @support https://www.boldt-webservice.de/en/forum-en/forum/bwtransifex.html
 * @license GNU/GPL, see LICENSE.txt
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

namespace BoldtWebservice\Component\BwTransifex\Administrator\Table;

// No direct access
defined('_JEXEC') or die('Restricted access');

use DateTime;
use Exception;
use Joomla\CMS\Factory;
use Joomla\CMS\Table\Table;
use Joomla\Database\DatabaseDriver;
use Joomla\Filter\InputFilter;

/**
 * Define the BwTransifex project table class
 *
 * @package BwTransifex Admin
 *
 * @since 1.0.0
 */
class ProjectTable extends Table
{
    /**
     * @var int Primary Key
     *
     * @since 1.0.0
     */
    public int $id = 0;

    /**
     * @var int asset_id
     *
     * @since 1.0.0
     */
    public $asset_id = 0;

    /**
     * @var string $title
     * varchar
     *
     * @since 1.0.0
     */
    public string $title = '';

    /**
     * @var string $transifex_slug
     * varchar
     *
     * @since 1.0.0
     */
    public string $transifex_slug = '';

    /**
     * @var string $description
     * text
     *
     * @since 1.0.0
     */
    public string $description = '';

    /**
     * @var string $alias
     *  varchar
     *
     * @since 1.0.0
     */
    public string $alias = '';

    /**
     * @var int $published
     *
     * @since 1.0.0
     */
    public int $published = 0;

    /**
     * @var datetime creation date of the project
     *
     * @since 1.0.0
     */
    public $created_date = '0000-00-00 00:00:00';

    /**
     * @var int user ID
     *
     * @since 1.0.0
     */
    public int $created_by = 0;

    /**
     * @var string
     *  varchar
     *
     * @since 1.0.0
     */
    public string $created_by_alias = '';

    /**
     * @var datetime|null last modification date of the project
     *
     * @since 1.0.0
     */
    public  $modified_time = null;

    /**
     * @var int user ID
     *
     * @since 1.0.0
     */
    public int $modified_by = 0;

    /**
     * @var int
     *
     * @since 1.0.0
     */
    public int $ordering = 0;

    /**
     * @var int access
     *
     * @since 1.0.0
     */
    public $access = 1;

    /**
     * @var string
     * mediumtext
     *
     * @since 1.0.0
     */
    public string $transifex_config = '';

    /**
     * @var string
     * varchar
     *
     * @since 1.0.0
     */
    public string $extension_name = '';

    /**
     * @var string
     * longtext
     *
     * @since 1.0.0
     */
    public $params = '';

    /**
     * Constructor
     *
     * @param   DatabaseDriver  $db  Database connector object
     *
     * @since 1.0.0
     */
    public function __construct($db)
    {
        parent::__construct('#__bwtransifex_projects', 'id', $db);

        $this->_jsonEncode = ['params'];
    }

    /**
     * Override Table::check method to ensure data integrity
     *
     * @access public
     *
     * @return boolean True
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function check(): bool
    {
        try {
            // Add event
            parent::check();

            // Sanitize values
            $filter = new InputFilter(array(), array(), 0, 0);

            $this->id                = $filter->clean($this->id, 'UINT');
            $this->asset_id          = $filter->clean($this->asset_id, 'UINT');
            $this->title             = trim(htmlspecialchars_decode($this->title, ENT_QUOTES));
            $this->transifex_slug    = $filter->clean(trim(htmlspecialchars_decode($this->transifex_slug, ENT_QUOTES)));
            $this->description       = $filter->clean($this->description, 'HTML');
            $this->alias             = trim(htmlspecialchars_decode($this->alias, ENT_QUOTES));
            $this->published         = $filter->clean($this->published, 'UINT');
            $this->created_time      = $filter->clean($this->created_time);
            $this->created_by        = $filter->clean($this->created_by, 'UINT');
            $this->created_by_alias  = $filter->clean($this->created_by_alias, 'STRING');
            $this->modified_time     = $filter->clean($this->modified_time);
            $this->modified_by       = $filter->clean($this->modified_by, 'UINT');
            $this->modified_by_alias = $filter->clean($this->modified_by_alias, 'STRING');
            $this->ordering          = $filter->clean($this->ordering, 'UINT');
            $this->access            = $filter->clean($this->access, 'UINT');
            $this->transifex_config  = $filter->clean($this->transifex_config, 'STRING');
            $this->extension_name    = $filter->clean($this->extension_name, 'STRING');
            $this->params            = $filter->clean($this->params, 'STRING');

            // Set alias
            if (trim($this->alias) == '')
            {
                $this->alias = str_replace(' ', '-', strtolower($this->title));
            }

            if (trim(str_replace('-', '', $this->alias)) == '')
            {
                $this->alias = Factory::getDate()->format('Y-m-d-H-i-s');
            }

            return true;
        }
        catch (Exception $exception)
        {
            throw new Exception($exception);
        }
    }

    /**
     * Override Table::store to set created/modified and user id.
     *
     * @param   boolean  $updateNulls  True to update fields even if they are null.
     *
     * @return  boolean  True on success.
     *
     * @throws Exception
     *
     * @since   1.0.0
     */
    public function store($updateNulls = false): bool
    {
        $date = Factory::getDate();
        $app  = Factory::getApplication();
        $user = $app->getIdentity();
        $id   = $this->id;

        if ($id)
        {
            // Existing project
            $this->modified_time = $date->toSql();
            $this->modified_by   = $user->id;
        }
        else
        {
            // New project
            $this->created_time = $date->toSql();
            $this->created_by   = $user->id;
        }

    return parent::store($updateNulls);
    }
}

