<?php
/**
 * BwTransifex Component
 *
 * BwTransifex resource model class for the component backend
 *
 * @version 1.0.1
 * @package BwTransifex
 * @subpackage BwTransifex Component Admin
 * @author Romana Boldt
 * @copyright (C) 2025 Boldt Webservice <forum@boldt-webservice.de>
 * @support https://www.boldt-webservice.de/en/forum-en/forum/bwtransifex.html
 * @license GNU/GPL, see LICENSE.txt
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

namespace BoldtWebservice\Component\BwTransifex\Administrator\Model;

use BoldtWebservice\Component\BwTransifex\Administrator\Helper\BwTransifexHelperPackage;
use Exception;
use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\BaseModel;

defined('_JEXEC') or die('Restricted access');

/**
 * Define the BwTransifex resource model class
 *
 * @package     BwTransifex Admin
 *
 * @since       1.0.0
 */
class ResourceModel extends BaseModel
{
    private int $projectId = 0;

    public function __construct(array $config = array()) {
        if(isset($config['project_id'])) {
            $this->projectId = $config['project_id'];
        }

        parent::__construct($config);
    }

    /**
     * Add a resource to the table
     *
     * @param   array  $resources
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function add(array $resources):void
    {
        // remove all resources for that project
        $this->cleanDb();

        // now add the resources
        $db = Factory::getContainer()->get('DatabaseDriver');

        foreach($resources as $resource) {
            $query = $db->getQuery(true);

            $query->insert($db->quoteName('#__bwtransifex_resources'));
            $query->columns(array(
                $db->quoteName('id'),
                $db->quoteName('project_id'),
                $db->quoteName('resource_name')
            ));
            $query->values(
                $db->quote(0) . ',' .
                $db->quote($this->projectId) . ',' .
                $db->quote($resource));

            $db->setQuery($query);
            $db->execute();
        }
    }

    /**
     * Clean up the resource table for a project
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    private function cleanDb() :void
    {
        $session = Factory::getApplication()->getSession();

        // Save used languages at session
        $languages = BwTransifexHelperPackage::getUsedLanguagesFromDb($this->projectId);
        $session->set('usedLanguages', $languages, 'bwtransifex');

        // Save state of previous old TX stats and store at session
        BwTransifexHelperPackage::storeLangStatsAtSession($this->projectId, 'previousTxUpdate');

        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);
        // remove the resources
        $query->delete('#__bwtransifex_resources');
        $query->where($db->qn('project_id') . '=' . $db->q($this->projectId));
        $db->setQuery($query);
        $db->execute();

        $query->clear();

        // remove all languages for the current project
        $query->delete('#__bwtransifex_languages');
        $query->where($db->qn('project_id') . '=' . $db->q($this->projectId));
        $db->setQuery($query);
        $db->execute();

        // Save state of previous language pack versions
        $previousLangPackVersions = array();
        foreach ($languages as $language)
        {
            $query->clear();
            $query->select($db->quoteName('project_id'));
            $query->select($db->quoteName('lang_name'));
            $query->select($db->quoteName('completed'));
            $query->select($db->quoteName('created_time'));
            $query->select($db->quoteName('lang_version'));
            $query->from($db->quoteName('#__bwtransifex_zips'));
            $query->where($db->quoteName('project_id') . ' = ' . $db->Quote($this->projectId));
            $query->where($db->quoteName('lang_name') . ' = ' . $db->Quote($language));

            $db->setQuery($query);

            $previousLangPackVersions[$language] = $db->loadAssoc();

        }

        $session->set('prevLangPackVersions', $previousLangPackVersions, 'bwtransifex');

        // remove all zips for the current project
        $query->clear();

        $query->delete('#__bwtransifex_zips')
            ->where($db->qn('project_id') . '=' . $db->q($this->projectId));
        $db->setQuery($query);
        $db->execute();
    }

}
