<?php
/**
 * BwTransifex Component
 *
 * BwTransifex lists model class for the component backend
 *
 * @version 1.0.1
 * @package BwTransifex
 * @subpackage BwTransifex Component Admin
 * @author Romana Boldt
 * @copyright (C) 2025 Boldt Webservice <forum@boldt-webservice.de>
 * @support https://www.boldt-webservice.de/en/forum-en/forum/bwtransifex.html
 * @license GNU/GPL, see LICENSE.txt
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

namespace BoldtWebservice\Component\BwTransifex\Administrator\Model;

// No direct access
defined('_JEXEC') or die('Restricted access');

use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\CMS\Table\Table;

/**
 * Define the BwTransifex projects model class
 *
 * @package BwTransifex Admin
 *
 * @since 1.0.0
 */
class ProjectsModel extends ListModel
{
    /**
     * The database object
     *
     * @var	?object
     *
     * @since       1.0.0
     */
    protected ?object $db = null;

    /**
     * The query object
     *
     * @var	object
     *
     * @since       1.0.0
     */
    protected $query;

    /**
     * Constructor
     *
     * @param   array  $config  An optional associative array of configuration settings.
     *
     * @throws Exception
     *
     * @since 1.0.0
     *
     */
    public function __construct($config = array())
    {
        if (empty($config['filter_fields']))
        {
            $config['filter_fields'] = array(
                'p.ordering',
                'p.published',
                'p.title',
                'p.access',
                'p.id',
            );
        }

        parent::__construct($config);
    }

    /**
     * Method to auto-populate the model state.
     *
     * Note. Calling getState in this method will result in recursion.
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    protected function populateState($ordering = null, $direction = null): void
    {
        // Initialise variables.
        $app = Factory::getApplication();

        // Load the filter state.
        $search = $this->getUserStateFromRequest($this->context . '.filter.search', 'filter_search', null, 'string');
        $this->setState('filter.search', $search);

        $filtersearch = $this->getUserStateFromRequest($this->context . '.filter.search_filter', 'filter_search_filter');
        $this->setState('filter.search_filter', $filtersearch);

        $state = $this->getUserStateFromRequest($this->context . '.filter.published', 'filter_published', '', 'string');
        $this->setState('filter.published', $state);

        $state = $this->getUserStateFromRequest($this->context . '.filter.access', 'filter_access', '', 'string');
        $this->setState('filter.access', $state);

        // Check if the ordering field is in the white list, otherwise use the incoming value.
        $filter_order = $app->getUserStateFromRequest($this->context . '.filter_order', 'filter_order', $ordering, 'string');

        if (!in_array($filter_order, $this->filter_fields))
        {
            $filter_order = $ordering;
            $app->setUserState($this->context . '.filter_order', $filter_order);
        }

        $this->setState('list.ordering', $filter_order);

        // List state information.
        parent::populateState('p.id', 'asc');

        $limitstart = $app->input->get->post->get('limitstart');
        $this->setState('list.start', $limitstart);

        // Load the parameters.
        $params = ComponentHelper::getParams('com_bwtransifex');
        $this->setState('params', $params);
    }

    /**
     * Method to get a store id based on model configuration state.
     *
     * This is necessary because the model is used by the component and
     * different modules that might need different sets of data or different
     * ordering requirements.
     *
     * @param	string		$id	A prefix for the store id.
     *
     * @return	string		A store id.
     *
     * @since	1.0.0
     */
    protected function getStoreId($id = ''): string
    {
        // Compile the store id.
        $id	.= ':' . $this->getState('filter.search');
        $id	.= ':' . $this->getState('filter.search_filter');
        $id	.= ':' . $this->getState('filter.parent_state');
        $id	.= ':' . $this->getState('filter.published');

        return parent::getStoreId($id);
    }

    /**
     * Returns a reference to a Table object, always creating it.
     *
     * @param   string  $name     The table type to instantiate
     * @param   string  $prefix   A prefix for the table class name. Optional.
     * @param   array   $options  Configuration array for model. Optional.
     *
     * @return        Table    A database object
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function getTable($name = 'Project', $prefix = 'Administrator', $options = array()): Table
    {
        return parent::getTable($name, $prefix, $options);
    }

    /**
     * Build an SQL query to load the list data.
     *
     * @throws Exception
     *
     * @since    1.0.0
     */
    protected function getListQuery()
    {
        // Create a new query object.
        $this->db    = Factory::getContainer()->get('DatabaseDriver');
        $this->query = $this->db->getQuery(true);

        // Select the required fields from the table.
        $this->query->select(
            'p.id, p.title, p.alias, p.description, ' .
            'p.published, p.access, p.created_time, p.created_by, p.ordering'
        );
        $this->query->from('#__bwtransifex_projects AS p');

        // Join over the asset groups.
        $this->query->select(' ag.title AS access_level');
        $this->query->join('LEFT', '#__viewlevels AS ag ON ag.id = p.access');

        $this->getQueryWhere();
        $this->getQueryOrder();

        return $this->query;
    }

    /**
     * Method to get an array of data items.
     *
     * @return  array  An array of data items on success, false on failure.
     *
     * @since   1.0.0
     */
    public function getItems():array
    {
        $items = parent::getItems();

        if (empty($items))
        {
            $items = array();
        }

        return $items;
    }

    /**
     * Method to build the MySQL query 'where' part
     *
     * @return 	void
     *
     * @throws Exception
     *
     * @since   1.0.0
     */
    private function getQueryWhere(): void
    {
        $this->getFilterByAccessLevelFilter();
        $this->getFilterByPublishedState();
        $this->getFilterBySearchword();
    }

    /**
     * Method to build the MySQL query 'order' part
     *
     * @return 	void
     *
     * @since   1.0.0
     */
    private function getQueryOrder(): void
    {
        $db        = $this->getDatabase();
        $orderCol  = $this->state->get('list.ordering', 'p.title');
        $orderDirn = $this->state->get('list.direction', 'asc');

        //sqlsrv change
        if ($orderCol == 'modified_time')
        {
            $orderCol = 'p.modified_time';
        }

        if ($orderCol == 'p.title')
        {
            $orderCol = 'p.title';
        }

        $this->query->order($db->quoteName($db->escape($orderCol)) . ' ' . $db->escape($orderDirn));
    }

    /**
     * Method to get the filter by access level
     *
     * @return 	void
     *
     * @throws Exception
     *
     * @since   1.0.0
     */
    private function getFilterByAccessLevelFilter(): void
    {
        $access = $this->getState('filter.access');

        if ($access)
        {
            $this->query->where($this->db->quoteName('p.access') . ' = ' . (int) $access);
        }
    }

    /**
     * Method to get the filter by published state
     *
     * @return 	void
     *
     * @since   1.0.0
     */
    private function getFilterByPublishedState(): void
    {
        $published = $this->getState('filter.published');

        if (is_numeric($published))
        {
            $this->query->where($this->db->quoteName('p.published') . ' = ' . (int) $published);
        }
        elseif ($published === '')
        {
            $this->query->where('(' . $this->db->quoteName('p.published') . ' = 0 OR ' . $this->db->quoteName('p.published') . ' = 1)');
        }
    }

    /**
     * Method to get the filter by search word
     *
     * @return 	void
     *
     * @since   1.0.0
     */
    private function getFilterBySearchword(): void
    {
//		$filtersearch = $this->getState('filter.search_filter');
        $rawSearch = $this->db->escape($this->getState('filter.search'), true);
        $search    = '%' . $rawSearch . '%';

        if (!empty($rawSearch))
        {
            $this->query->where($this->db->quoteName('p.title') . ' LIKE ' . $this->db->quote($search, false));
        }
    }
}
