<?php
/**
 * BwTransifex Component
 *
 * BwTransifex language helper class for the component backend
 *
 * This is a fork of the Text_LanguageDetect class from the PEAR project Text_LanguageDetect
 *
 * @version 1.0.1
 * @package BwTransifex
 * @subpackage BwTransifex Component Admin
 * @author Romana Boldt
 * @copyright (C) 2025 Boldt Webservice <forum@boldt-webservice.de>
 * @support https://www.boldt-webservice.de/en/forum-en/forum/bwtransifex.html
 * @license GNU/GPL, see LICENSE.txt
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 */

/**
 * Define the BwTransifex language helper class
 *
 * Part of Text_LanguageDetect
 *
 * PHP version 5
 *
 * @category  Text
 * @package   Text_LanguageDetect
 * @author    Christian Weiske <cweiske@php.net>
 * @copyright 2011 Christian Weiske <cweiske@php.net>
 * @license   https://www.debian.org/misc/bsd.license BSD
 * @version   SVN: $Id$
 * @link      https://pear.php.net/package/Text_LanguageDetect/
 */

/**
 * Provides a mapping between the languages from lang.dat and the
 * ISO 639-1 and ISO-639-2 codes.
 *
 * Note that this class contains only languages that exist in lang.dat.
 *
 * @category  Text
 * @package   Text_LanguageDetect
 * @author    Christian Weiske <cweiske@php.net>
 * @copyright 2011 Christian Weiske <cweiske@php.net>
 * @license   https://www.debian.org/misc/bsd.license BSD
 * @link      https://www.loc.gov/standards/iso639-2/php/code_list.php
 */

namespace BoldtWebservice\Component\BwTransifex\Administrator\Helper;

defined('_JEXEC') or die('Restricted access');

/**
 * Define the BwTransifex language helper class
 *
 * @package BwTransifex Admin
 *
 * @since   1.0.0
 */
class BwTransifexHelperLanguage
{
    /**
     * Maps all language names from the language database to the
     * ISO 639-1 2-letter language code.
     *
     * NULL indicates that there is no 2-letter code.
     *
     * @var array
     *
     * @since 1.0.0
     */
    public static array $nameToCode2 = array(
        'africaans' => 'af',
        'albanian' => 'sq',
        'amharic' => 'am',
        'arabic' => 'ar',
        'azeri' => 'az',
        'belarusian' => 'bel',
        'bengali' => 'bn',
        'bulgarian' => 'bg',
        'cebuano' => null,
        'chinese' => 'zh',
        'croatian' => 'hr',
        'czech' => 'cs',
        'danish' => 'da',
        'dutch' => 'nl',
        'english' => 'en',
        'esperanto' => 'eo',
        'estonian' => 'et',
        'farsi' => 'fa',
        'finnish' => 'fi',
        'french' => 'fr',
        'german' => 'de',
        'hausa' => 'ha',
        'hawaiian' => null,
        'hindi' => 'hi',
        'hungarian' => 'hu',
        'icelandic' => 'is',
        'indonesian' => 'id',
        'italian' => 'it',
        'kazakh' => 'kk',
        'kurdish' => 'ku',
        'kyrgyz' => 'ky',
        'latin' => 'la',
        'latvian' => 'lv',
        'lithuanian' => 'lt',
        'macedonian' => 'mk',
        'mongolian' => 'mn',
        'nepali' => 'ne',
        'norwegian' => 'no',
        'pashto' => 'ps',
        'pidgin' => null,
        'polish' => 'pl',
        'portuguese' => 'pt',
        'romanian' => 'ro',
        'russian' => 'ru',
        'serbian' => 'sr',
        'slovak' => 'sk',
        'slovene' => 'sl',
        'somali' => 'so',
        'spanish' => 'es',
        'swahili' => 'sw',
        'swedish' => 'sv',
        'tagalog' => 'tl',
        'turkish' => 'tr',
        'ukrainian' => 'uk',
        'urdu' => 'ur',
        'uzbek' => 'uz',
        'vietnamese' => 'vi',
        'welsh' => 'cy',

    );

    /**
     * Maps all language names from the language database to the
     * ISO 639-2 3-letter language code.
     *
     * @var array
     *
     * @since 1.0.0
     */
    public static array $nameToCode3 = array(
        'albanian' => 'sqi',
        'amharic' => 'amh',
        'arabic' => 'ara',
        'azeri' => 'aze',
        'belarusian' => 'bel',
        'bengali' => 'ben',
        'bulgarian' => 'bul',
        'cebuano' => 'ceb',
        'croatian' => 'hrv',
        'czech' => 'ces',
        'danish' => 'dan',
        'dutch' => 'nld',
        'english' => 'eng',
        'estonian' => 'est',
        'farsi' => 'fas',
        'finnish' => 'fin',
        'french' => 'fra',
        'german' => 'deu',
        'hausa' => 'hau',
        'hawaiian' => 'haw',
        'hindi' => 'hin',
        'hungarian' => 'hun',
        'icelandic' => 'isl',
        'indonesian' => 'ind',
        'italian' => 'ita',
        'kazakh' => 'kaz',
        'kyrgyz' => 'kir',
        'latin' => 'lat',
        'latvian' => 'lav',
        'lithuanian' => 'lit',
        'macedonian' => 'mkd',
        'mongolian' => 'mon',
        'nepali' => 'nep',
        'norwegian' => 'nor',
        'pashto' => 'pus',
        'pidgin' => 'crp',
        'polish' => 'pol',
        'portuguese' => 'por',
        'romanian' => 'ron',
        'russian' => 'rus',
        'serbian' => 'srp',
        'slovak' => 'slk',
        'slovene' => 'slv',
        'somali' => 'som',
        'spanish' => 'spa',
        'swahili' => 'swa',
        'swedish' => 'swe',
        'tagalog' => 'tgl',
        'turkish' => 'tur',
        'ukrainian' => 'ukr',
        'urdu' => 'urd',
        'uzbek' => 'uzb',
        'vietnamese' => 'vie',
        'welsh' => 'cym',
    );

    /**
     * Maps ISO 639-1 2-letter language codes to the language names
     * in the language database
     *
     * Not all languages have a 2-letter code, so some are missing
     *
     * @var array
     *
     * @since 1.0.0
     */
    public static array $code2ToName = array(
        'af' => 'africaans',
        'am' => 'amharic',
        'ar' => 'arabic',
        'az' => 'azeri',
        'be' => 'belarusian',
        'bg' => 'bulgarian',
        'bn' => 'bengali',
        'bs' => 'bosnian',
        'ca' => 'catalan',
        'cs' => 'czech',
        'cy' => 'welsh',
        'da' => 'danish',
        'de' => 'german',
        'el' => 'greek',
        'en' => 'english',
        'eo' => 'esperanto',
        'es' => 'spanish',
        'et' => 'estonian',
        'gl' => 'galician',
        'gu' => 'gujarati',
        'ja' => 'japanese',
        'fa' => 'farsi',
        'fi' => 'finnish',
        'fr' => 'french',
        'ha' => 'hausa',
        'he' => 'hebrew',
        'hi' => 'hindi',
        'hr' => 'croatian',
        'hu' => 'hungarian',
        'hy' => 'armenian',
        'id' => 'indonesian',
        'is' => 'icelandic',
        'it' => 'italian',
        'ka' => 'Georgian',
        'kk' => 'kazakh',
        'km' => 'Khmer',
        'ko' => 'korean',
        'ku' => 'kurdish',
        'ky' => 'kyrgyz',
        'la' => 'latin',
        'lt' => 'lithuanian',
        'lv' => 'latvian',
        'mk' => 'macedonian',
        'mn' => 'mongolian',
        'mr' => 'marathi',
        'ms' => 'malay',
        'my' => 'burmese',
        'nb' => 'norwegian',
        'ne' => 'nepali',
        'nl' => 'dutch',
        'no' => 'norwegian',
        'pl' => 'polish',
        'ps' => 'pashto',
        'pt' => 'portuguese',
        'ro' => 'romanian',
        'ru' => 'russian',
        'sk' => 'slovak',
        'sl' => 'slovene',
        'so' => 'somali',
        'sq' => 'albanian',
        'sr' => 'serbian',
        'sv' => 'swedish',
        'sw' => 'swahili',
        'ta' => 'tamil',
        'te' => 'telugo',
        'th' => 'thai',
        'tl' => 'tagalog',
        'tr' => 'turkish',
        'ug' => 'uighur',
        'uk' => 'ukrainian',
        'ur' => 'urdu',
        'uz' => 'uzbek',
        'vi' => 'vietnamese',
        'zh' => 'chinese',
    );

    /**
     * Maps ISO 639-2 3-letter language codes to the language names
     * in the language database.
     *
     * @var array
     *
     * @since 1.0.0
     */
    public static array $code3ToName = array(
        'ara' => 'arabic',
        'aze' => 'azeri',
        'bel' => 'belarusian',
        'ben' => 'bengali',
        'bul' => 'bulgarian',
        'ceb' => 'cebuano',
        'ces' => 'czech',
        'crp' => 'pidgin',
        'cym' => 'welsh',
        'dan' => 'danish',
        'deu' => 'german',
        'eng' => 'english',
        'est' => 'estonian',
        'fas' => 'farsi',
        'fin' => 'finnish',
        'fra' => 'french',
        'hau' => 'hausa',
        'haw' => 'hawaiian',
        'hin' => 'hindi',
        'hrv' => 'croatian',
        'hun' => 'hungarian',
        'ind' => 'indonesian',
        'isl' => 'icelandic',
        'ita' => 'italian',
        'kaz' => 'kazakh',
        'kir' => 'kyrgyz',
        'lat' => 'latin',
        'lav' => 'latvian',
        'lit' => 'lithuanian',
        'mkd' => 'macedonian',
        'mon' => 'mongolian',
        'nep' => 'nepali',
        'nld' => 'dutch',
        'nor' => 'norwegian',
        'pol' => 'polish',
        'por' => 'portuguese',
        'pus' => 'pashto',
        'rom' => 'romanian',
        'rus' => 'russian',
        'slk' => 'slovak',
        'slv' => 'slovene',
        'som' => 'somali',
        'spa' => 'spanish',
        'sqi' => 'albanian',
        'srp' => 'serbian',
        'swa' => 'swahili',
        'swe' => 'swedish',
        'tgl' => 'tagalog',
        'tur' => 'turkish',
        'ukr' => 'ukrainian',
        'urd' => 'urdu',
        'uzb' => 'uzbek',
        'vie' => 'vietnamese',
    );

    public static array $code2ToCountry = array(
        'aa' => 'Unitag',
        'af' => 'Afghanistan',
        'ax' => 'Åland Islands',
        'al' => 'Albania',
        'dz' => 'Algeria',
        'as' => 'American Samoa',
        'ad' => 'Andorra',
        'ao' => 'Angola',
        'ai' => 'Anguilla',
        'aq' => 'Antarctica',
        'ag' => 'Antigua and Barbuda',
        'ar' => 'Argentina',
        'am' => 'Armenia',
        'aw' => 'Aruba',
        'au' => 'Australia',
        'at' => 'Austria',
        'az' => 'Azerbaijan',
        'bs' => 'Bahamas',
        'bh' => 'Bahrain',
        'bd' => 'Bangladesh',
        'bb' => 'Barbados',
        'by' => 'Belarus',
        'be' => 'Belgium',
        'bz' => 'Belize',
        'bj' => 'Benin',
        'bm' => 'Bermuda',
        'bt' => 'Bhutan',
        'bo' => 'Bolivia',
        'bq' => 'Bonaire',
        'ba' => 'Bosnia and Herzegovina',
        'bw' => 'Botswana',
        'bv' => 'Bouvet Island',
        'br' => 'Brazil',
        'io' => 'British Indian Ocean Territory',
        'bn' => 'Brunei Darussalam',
        'bg' => 'Bulgaria',
        'bf' => 'Burkina Faso',
        'bi' => 'Burundi',
        'kh' => 'Cambodia',
        'cm' => 'Cameroon',
        'ca' => 'Canada',
        'cv' => 'Cape Verde',
        'ky' => 'Cayman Islands',
        'cf' => 'Central African Republic',
        'td' => 'Chad',
        'cl' => 'Chile',
        'cn' => 'China',
        'cx' => 'Christmas Island',
        'cc' => 'Cocos (keeling) Islands',
        'co' => 'Colombia',
        'km' => 'Comoros',
        'cg' => 'Congo',
        'cd' => 'Congo',
        'ck' => 'Cook Islands',
        'cr' => 'Costa rica',
        'ci' => 'Côte d\'ivoire',
        'hr' => 'Croatia',
        'cu' => 'Cuba',
        'cw' => 'Curaçao',
        'cy' => 'Cyprus',
        'cz' => 'Czech Republic',
        'dk' => 'Denmark',
        'dj' => 'Djibouti',
        'dm' => 'Dominica',
        'do' => 'Dominican Republic',
        'ec' => 'Ecuador',
        'eg' => 'Egypt',
        'sv' => 'El Salvador',
        'gq' => 'Equatorial Guinea',
        'er' => 'Eritrea',
        'ee' => 'Estonia',
        'et' => 'Ethiopia',
        'fk' => 'Falkland Islands (Malvinas)',
        'fo' => 'Faroe Islands',
        'fj' => 'Fiji',
        'fi' => 'Finland',
        'fr' => 'France',
        'gf' => 'French Guiana',
        'pf' => 'French Polynesia',
        'tf' => 'French Southern Territories',
        'ga' => 'Gabon',
        'gm' => 'Gambia',
        'ge' => 'Georgia',
        'de' => 'Germany',
        'gh' => 'Ghana',
        'gi' => 'Gibraltar',
        'gr' => 'Greece',
        'gl' => 'Greenland',
        'gd' => 'Grenada',
        'gp' => 'Guadeloupe',
        'gu' => 'Guam',
        'gt' => 'Guatemala',
        'gg' => 'Guernsey',
        'gn' => 'Guinea',
        'gw' => 'Guinea-Bissau',
        'gy' => 'Guyana',
        'ht' => 'Haiti',
        'hm' => 'Heard Island and McDonald Islands',
        'va' => 'Holy See (Vatican City State)',
        'hn' => 'Honduras',
        'hk' => 'Hong Kong',
        'hu' => 'Hungary',
        'is' => 'Iceland',
        'in' => 'India',
        'id' => 'Indonesia',
        'ir' => 'Iran, Islamic Republic of',
        'iq' => 'Iraq',
        'ie' => 'Ireland',
        'im' => 'Isle of Man',
        'il' => 'Israel',
        'it' => 'Italy',
        'jm' => 'Jamaica',
        'jp' => 'Japan',
        'je' => 'Jersey',
        'jo' => 'Jordan',
        'kz' => 'Kazakhstan',
        'ke' => 'Kenya',
        'ki' => 'Kiribati',
        'kp' => 'Korea, Democratic People\'s Republic of',
        'kr' => 'Korea, Republic of',
        'kw' => 'Kuwait',
        'kg' => 'Kyrgyzstan',
        'la' => 'Lao People\'s Democratic Republic',
        'lv' => 'Latvia',
        'lb' => 'Lebanon',
        'ls' => 'Lesotho',
        'lr' => 'Liberia',
        'ly' => 'Libya',
        'li' => 'Liechtenstein',
        'lt' => 'Lithuania',
        'lu' => 'Luxembourg',
        'mo' => 'Macao',
        'mk' => 'Macedonia',
        'mg' => 'Madagascar',
        'mw' => 'Malawi',
        'my' => 'Malaysia',
        'mv' => 'Maldives',
        'ml' => 'Mali',
        'mt' => 'Malta',
        'mh' => 'Marshall Islands',
        'mq' => 'Martinique',
        'mr' => 'Mauritania',
        'mu' => 'Mauritius',
        'yt' => 'Mayotte',
        'mx' => 'Mexico',
        'fm' => 'Micronesia',
        'md' => 'Moldova, Republic of',
        'mc' => 'Monaco',
        'mn' => 'Mongolia',
        'me' => 'Montenegro',
        'ms' => 'Montserrat',
        'ma' => 'Morocco',
        'mz' => 'Mozambique',
        'mm' => 'Myanmar',
        'na' => 'Namibia',
        'nr' => 'Nauru',
        'np' => 'Nepal',
        'nl' => 'Netherlands',
        'nc' => 'New Caledonia',
        'nz' => 'New Zealand',
        'ni' => 'Nicaragua',
        'ne' => 'Niger',
        'ng' => 'Nigeria',
        'nu' => 'Niue',
        'nf' => 'Norfolk Island',
        'mp' => 'Northern Mariana Islands',
        'no' => 'Norway',
        'om' => 'Oman',
        'pk' => 'Pakistan',
        'pw' => 'Palau',
        'ps' => 'Palestinian Territory, occupied',
        'pa' => 'Panama',
        'pg' => 'Papua New Guinea',
        'py' => 'Paraguay',
        'pe' => 'Peru',
        'ph' => 'Philippines',
        'pn' => 'Pitcairn',
        'pl' => 'Poland',
        'pt' => 'Portugal',
        'pr' => 'Puerto Rico',
        'qa' => 'Qatar',
        're' => 'Réunion',
        'ro' => 'Romania',
        'ru' => 'Russian Federation',
        'rw' => 'Rwanda',
        'bl' => 'Saint Barthélemy',
        'sh' => 'Saint Helena, Ascension and Tristan da Cunha',
        'kn' => 'Saint Kitts and Nevis',
        'lc' => 'Saint Lucia',
        'mf' => 'Saint Martin (french part)',
        'pm' => 'Saint Pierre and Miquelon',
        'vc' => 'Saint Vincent and the Grenadines',
        'ws' => 'Samoa',
        'sm' => 'San Marino',
        'st' => 'Sao Tome and Principe',
        'sa' => 'Saudi Arabia',
        'sn' => 'Senegal',
        'rs' => 'Serbia',
        'sc' => 'Seychelles',
        'sl' => 'Sierra Leone',
        'sg' => 'Singapore',
        'sx' => 'Sint Maarten (dutch part)',
        'sk' => 'Slovakia',
        'si' => 'Slovenia',
        'sb' => 'Solomon Islands',
        'so' => 'Somalia',
        'gs' => 'South Georgia and the South Sandwich Islands',
        'ss' => 'South Sudan',
        'es' => 'Spain',
        'lk' => 'Sri Lanka',
        'sd' => 'Sudan',
        'sr' => 'Suriname',
        'sj' => 'Svalbard and Jan Mayen',
        'sz' => 'Swaziland',
        'se' => 'Sweden',
        'ch' => 'Switzerland',
        'sy' => 'Syrian Arab Republic',
        'tw' => 'Taiwan, Province of China',
        'tj' => 'Tajikistan',
        'tz' => 'Tanzania, United Republic of',
        'th' => 'Thailand',
        'tl' => 'Timor-Leste',
        'tg' => 'Togo',
        'tk' => 'Tokelau',
        'to' => 'Tonga',
        'tt' => 'Trinidad and Tobago',
        'tn' => 'Tunisia',
        'tr' => 'Turkey',
        'tm' => 'Turkmenistan',
        'tc' => 'Turks and Caicos Islands',
        'tv' => 'Tuvalu',
        'ug' => 'Uganda',
        'ua' => 'Ukraine',
        'ae' => 'United Arab Emirates',
        'gb' => 'United Kingdom',
        'us' => 'United States',
        'um' => 'United States Minor Outlying Islands',
        'uy' => 'Uruguay',
        'uz' => 'Uzbekistan',
        'vu' => 'Vanuatu',
        've' => 'Venezuela, Bolivarian Republic of',
        'vn' => 'Viet Nam',
        'vg' => 'Virgin Islands, british',
        'vi' => 'Virgin Islands, u.s.',
        'wf' => 'Wallis and Futuna',
        'eh' => 'Western Sahara',
        'xx' => '',
        'ye' => 'Yemen',
        // There is no latin country, but transifex has named the lang so...
        'yu' => 'Latin',
        'za' => 'South Africa',
        'zm' => 'Zambia',
        'zw' => 'Zimbabwe',
    );

    /**
     * Returns the 2-letter ISO 639-1 code for the given language name.
     *
     * @param string $lang English language name like "swedish"
     *
     * @return ?string Two-letter language code (e.g. "sv") or NULL if not found
     *
     * @since 1.0.0
     */
    public static function nameToCode2(string $lang): ?string
    {
        $lang = strtolower($lang);

        if (!isset(self::$nameToCode2[$lang]))
        {
            return null;
        }

        return self::$nameToCode2[$lang];
    }

    /**
     * Returns the 3-letter ISO 639-2 code for the given language name.
     *
     * @param string $lang English language name like "swedish"
     *
     * @return ?string Three-letter language code (e.g. "swe") or NULL if not found
     *
     * @since 1.0.0
     */
    public static function nameToCode3(string $lang): ?string
    {
        $lang = strtolower($lang);

        if (!isset(self::$nameToCode3[$lang]))
        {
            return null;
        }

        return self::$nameToCode3[$lang];
    }

    /**
     * Returns the language name for the given 2-letter ISO 639-1 code.
     *
     * @param string $code Two-letter language code (e.g. "sv")
     *
     * @return ?string English language name like "swedish"
     *
     * @since 1.0.0
     */
    public static function code2ToName(string $code): ?string
    {
        $lang = strtolower($code);

        if (!isset(self::$code2ToName[$lang]))
        {
            return null;
        }

        return self::$code2ToName[$lang];
    }

    /**
     * Returns the language name for the given 3-letter ISO 639-2 code.
     *
     * @param string $code Three-letter language code (e.g. "swe")
     *
     * @return ?string English language name like "swedish"
     *
     * @since 1.0.0
     */
    public static function code3ToName(string $code): ?string
    {
        $lang = strtolower($code);

        if (!isset(self::$code3ToName[$lang]))
        {
            return null;
        }

        return self::$code3ToName[$lang];
    }

    /**
     * Returns the country name for the given 2-letter ISO 3166-1-alpha-2 code.
     *
     * @param   ?string  $code  Two-letter language code (e.g. "bg")
     *
     * @return string|null English language name like "swedish"
     *
     * @since 1.0.0
     */
    public static function code2ToCountry(?string $code): ?string
    {
        if (is_null($code))
        {
            return null;
        }

        $lang = strtolower($code);

        if (!isset(self::$code2ToCountry[$lang]))
        {
            return null;
        }

        return self::$code2ToCountry[$lang];
    }
}
